<?php namespace App\Controllers\Api\Customer\Payment\Paystack;

use App\Controllers\PrivateController;
use App\Models\AppsModel;
use App\Models\DepositMethodsModel;
use App\Models\PlansModel;
use App\Models\SubscribesModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Database\Exceptions\DatabaseException;

class CreatePaymentRequest extends PrivateController
{
    protected $paystackUrl = 'https://api.paystack.co/transaction/initialize';
    protected $method;

    public function __construct()
    {
        // Không gọi parent::__construct() nếu lớp cha không có phương thức khởi tạo
        $depositMethods = new DepositMethodsModel();
        $this->method = $depositMethods->where([
            'name' => 'Paystack',
            'status' => 1
        ])->first();

        if (!$this->method) {
            throw new DatabaseException('Paystack method not found or inactive.');
        }
    }

    public function index(): ResponseInterface
    {
        // Kiểm tra dữ liệu đầu vào
        if (!$this->validate($this->create_validation_type())) {
            return $this->respond(["message" => $this->validator->getErrors()], 400);
        }

        $uid = esc($this->request->getGet("uid"));

        $db = \Config\Database::connect();
        $db->transStart();

        try {
            $app = $this->getApp($uid);
            $this->checkActiveSubscription($app["id"]);
            $plan = $this->getPlan();

            $payment = $this->createPaystackPayment($plan, $app["id"]);

            if (isset($payment['data']['authorization_url'])) {
                $db->transComplete();
                return $this->respond(["url" => $payment['data']['authorization_url']], 200);
            } else {
                throw new DatabaseException("Paystack payment creation failed.");
            }
        } catch (\Exception $e) {
            $db->transRollback();
            return $this->respond(["message" => $e->getMessage()], 400);
        }
    }

    private function getApp($uid)
    {
        $projects = new AppsModel();
        $app = $projects->where([
            "uid" => $uid,
            "user" => $this->userId,
            "deleted_at" => 0
        ])->select("id")->first();

        if (!$app) {
            throw new DatabaseException(lang("Message.message_14"));
        }

        return $app;
    }

    private function checkActiveSubscription($appId)
    {
        $subscribes = new SubscribesModel();
        $is_active = $subscribes->where([
            "app_id" => $appId,
            "expires_at >" => time(),
            "is_disable" => 0
        ])->countAllResults();

        if ($is_active) {
            throw new DatabaseException(lang("Message.message_86"));
        }
    }

    private function getPlan()
    {
        $plans = new PlansModel();
        $plan = $plans->where([
            "id" => (int)$this->request->getJsonVar("plan_id"),
            "status" => 1,
            "deleted_at" => 0
        ])->first();

        if (!$plan) {
            throw new DatabaseException(lang("Message.message_83"));
        }

        return $plan;
    }

    private function createPaystackPayment($plan, $appId)
    {
        $settings = new \App\Libraries\Settings();
        $frontUrl = rtrim($settings->get_config("site_url"), '/') . '/';

        $returnUrl = base_url('public/ipn/paystack/capture');
        $cancelUrl = $frontUrl . 'private/profile/subscribe';

        $secretKey = $this->method["api_value_2"]; // Paystack Secret Key

        // Lấy mã tiền tệ từ bảng settings
        $currencyCode = $settings->get_config("currency_code");

        log_message('debug', 'Paystack Secret Key: ' . $secretKey);
        log_message('debug', 'Currency Code: ' . $currencyCode);

        $client = \Config\Services::curlrequest([
            'base_uri' => $this->paystackUrl,
            'headers' => [
                'Authorization' => "Bearer $secretKey",
                'Content-Type' => 'application/json'
            ]
        ]);

        $response = $client->post('', [
            'json' => [
                'amount' => $plan["price"] * 100, // Paystack yêu cầu số tiền là integer (kobo)
                'email' => $this->getUserEmail(),
                'callback_url' => $returnUrl,
                'metadata' => [
                    'custom_id' => $plan["id"] . "_" .  $this->userId . "_" . $appId
                ]
            ]
        ]);

        return json_decode($response->getBody(), true);
    }

    private function getUserEmail()
    {
        $users = new \App\Models\UsersModel();
        $user = $users->where('id', $this->userId)->select('email')->first();
        if (!$user) {
            throw new \Exception("User not found.");
        }
        return $user['email'];
    }

    private function create_validation_type(): array
    {
        return [
            "plan_id" => [
                "label" => lang("Fields.field_148"),
                "rules" => "required|numeric"
            ],
        ];
    }
}
